!===============================================================================
! Copyright (C) 2021 Intel Corporation
!
! This software and the related documents are Intel copyrighted  materials,  and
! your use of  them is  governed by the  express license  under which  they were
! provided to you (License).  Unless the License provides otherwise, you may not
! use, modify, copy, publish, distribute,  disclose or transmit this software or
! the related documents without Intel's prior written permission.
!
! This software and the related documents  are provided as  is,  with no express
! or implied  warranties,  other  than those  that are  expressly stated  in the
! License.
!===============================================================================

!
!   Content : Intel(R) oneAPI Math Kernel Library (Intel(R) oneMKL) Sparse BLAS
!             Fortran-2003 example
!
!*******************************************************************************
!
! Example program for using  Intel oneMKL Sparse BLAS Level 2 and 3
! for matrices represented in the compressed sparse column (CSC) storage scheme.
! The following Sparse Blas routines are used in the example:
!
!    Initialization:
!          MKL_SPARSE_D_CREATE_CSC     MKL_SPARSE_DESTROY
!
!    Inspection:
!          MKL_SPARSE_SET_MV_HINT      MKL_SPARSE_SET_SV_HINT
!          MKL_SPARSE_SET_MM_HINT      MKL_SPARSE_SET_SM_HINT
!          MKL_SPARSE_OPTIMIZE
!
!    Execution:
!          MKL_SPARSE_D_MV             MKL_SPARSE_D_TRSV
!          MKL_SPARSE_D_MM             MKL_SPARSE_D_TRSM
!
! Consider the matrix A (see Appendix 'Sparse Storage Formats for Sparse Blas
! level 2-3')
!
!                 |   1   -1      0   -3   |
!                 |  -2    5      0    0   | 
!   A    =        |   0    0      4    6   |
!                 |   0    0      2    7   |
!
! decomposed as
!
!                      A = L + D + U,
!
!  where L is the strict lower triangle of A, U is the strictly upper triangle
!  of A, D is the main diagonal.

!  The matrix A is represented in the compressed sparse column (CSC) storage scheme
!  with the help of three arrays (see Appendix 'Sparse Matrix Storage') as follows:
!  Using the native column major format for each dense block, we have the three
!  arrays:
!
!         colPtr = (  1     3   5    7     10 )
!         rowIndx = ( 1  2  1 2 3 4  1 3 4 )
!         values  = ( 1 -2 -1 5 4 2 -3 6 7 )

!  In what follows the symbol ' means taking of transposed.
!
!  The test performs the following operations :
!
!  Task 1. The example computes (L+D)'*S = F using MKL_SPARSE_D_MM where S is a known 4 by 2
!          matrix and then the example solves the system (L+D)'*X = F with the help of
!          MKL_SPARSE_D_TRSM. It is evident that X should be equal to S.
!
!  Task 2. The example computes (U+I)*S = F using MKL_SPARSE_D_MV where S is a vector
!          and then the example calls MKL_SPARSE_D_TRSV solves the system (U+I)*X = F 
!          with the single right hand side. It's evident that X should be equal to S.
!
!  Task 3. The next step is the computation (L+D+L') S = F using MKL_SPARSE_D_MV where S is
!          a vector. It is easy to see that L+D+L' is a symmetric matrix.
!
!  Task 4. The next step is the computation F = beta F + alpha A'* S using MKL_SPARSE_D_MV 
!          where S is a vector.
!
!*******************************************************************************
!     Definition arrays for sparse representation of the matrix A in
!     the sparse compresse column sparse format:
!*******************************************************************************
PROGRAM SPARSE_D_CSR_EXAMPLE
    USE MKL_SPBLAS
    USE ISO_C_BINDING
    IMPLICIT NONE

    INTEGER m, nnz, ldx, ldy
    INTEGER, ALLOCATABLE :: rowIndx(:), colPtr(:)
    DOUBLE PRECISION, ALLOCATABLE :: values(:)
    DOUBLE PRECISION, ALLOCATABLE :: rhs(:,:), sol(:,:), temp(:,:)
    DOUBLE PRECISION alpha, beta
    INTEGER nrhs, i, j, info, exit_status
    TYPE(SPARSE_MATRIX_T) cscA
    TYPE(MATRIX_DESCR) descrA
!*******************************************************************************
!    Instantiation of local variables :
!*******************************************************************************

    m = 4    ! matrix is 4 x 4
    nnz = 9  ! total number of nonzeros is 9
    nrhs = 2 ! number of right hand sides
    exit_status = 0

    ! matrix data
    ALLOCATE(colPtr(m+1))
    ALLOCATE(rowIndx(nnz))
    ALLOCATE(values(nnz))
    
    colPtr  = (/   1,          3,        5,         7,           10 /)
    rowIndx = (/   1,    2,    1,   2,   3,   4,    1,   3,   4 /)
    values  = (/ 1.0, -2.0, -1.0, 5.0, 4.0, 2.0, -3.0, 6.0, 7.0 /)

    ! rhs vector/matrix data
    ALLOCATE(rhs(m, nrhs))
    ALLOCATE(sol(m, nrhs))
    ALLOCATE(temp(m, nrhs))

    sol = reshape((/ 1.0, 1.0, 1.0, 1.0, 4.0, 3.0, 2.0, 1.0 /), (/ m,nrhs /))

    ! scalars
    alpha = 1.0
    beta = 0.0
    
    print*
    print 100
    print*, ' EXAMPLE PROGRAM FOR                                '
    print*, ' SPARSE COMPRESSED SPARSE ROW (CSC) FORMAT ROUTINES '
    print 100
    
!   Create CSC matrix
    info = MKL_SPARSE_D_CREATE_CSC(cscA, SPARSE_INDEX_BASE_ONE, m, m, colPtr, colPtr(2), rowIndx, values)
    IF (info .NE. 0) THEN
        print *, '  MKL_SPARSE_D_CREATE_CSC: '
        print 106, info
        exit_status = 1
        GOTO 99
    END IF

!*******************************************************************************
! Note that when setting up hints, some operations will have no internal 
! optimizations available and in this case, will return 
! SPARSE_STATUS_NOT_SUPPORTED(=6) so either this or SPARSE_STATUS_SUCCESS(=0) 
! are acceptable outputs, anything else should be considered as an error and 
! exit the program.
!*******************************************************************************


!*******************************************************************************
!  Setup hint for Task 1: lower triangular non-unit transpose MM/TRSM
!*******************************************************************************

    descrA % TYPE = SPARSE_MATRIX_TYPE_TRIANGULAR
    descrA % MODE = SPARSE_FILL_MODE_LOWER
    descrA % DIAG = SPARSE_DIAG_NON_UNIT

    info = MKL_SPARSE_SET_MM_HINT(cscA, SPARSE_OPERATION_TRANSPOSE, descrA, SPARSE_LAYOUT_COLUMN_MAJOR, nrhs, 1)
    IF ( (info.NE.SPARSE_STATUS_SUCCESS).AND.(info.NE.SPARSE_STATUS_NOT_SUPPORTED) ) THEN
        print *, '  Task 1 Hint: MKL_SPARSE_SET_MM_HINT: '
        print 106, info
        exit_status = 1
        GOTO 99
    END IF

    info = MKL_SPARSE_SET_SM_HINT(cscA, SPARSE_OPERATION_TRANSPOSE, descrA, SPARSE_LAYOUT_COLUMN_MAJOR, nrhs, 1)
    IF ( (info.NE.SPARSE_STATUS_SUCCESS).AND.(info.NE.SPARSE_STATUS_NOT_SUPPORTED) ) THEN
        print *, '  Task 1 Hint: MKL_SPARSE_SET_SM_HINT: '
        print 106, info
        exit_status = 1
        GOTO 99
    END IF

!*******************************************************************************
!  Setup hint for Task 2: Upper triangular unit MV/TRSV
!*******************************************************************************

    descrA % TYPE = SPARSE_MATRIX_TYPE_TRIANGULAR
    descrA % MODE = SPARSE_FILL_MODE_UPPER
    descrA % DIAG = SPARSE_DIAG_UNIT

    info = MKL_SPARSE_SET_MV_HINT(cscA, SPARSE_OPERATION_NON_TRANSPOSE, descrA, 1)
    IF ( (info.NE.SPARSE_STATUS_SUCCESS).AND.(info.NE.SPARSE_STATUS_NOT_SUPPORTED) ) THEN
        print *, '  Task 2 Hint: MKL_SPARSE_SET_MV_HINT: '
        print 106, info
        exit_status = 1
        GOTO 99
    END IF

    info = MKL_SPARSE_SET_SV_HINT(cscA, SPARSE_OPERATION_NON_TRANSPOSE, descrA, 1)
    IF ( (info.NE.SPARSE_STATUS_SUCCESS).AND.(info.NE.SPARSE_STATUS_NOT_SUPPORTED) ) THEN
        print *, '  Task 2 Hint: MKL_SPARSE_SET_SV_HINT: '
        print 106, info
        exit_status = 1
        GOTO 99
    END IF

!*******************************************************************************
!  Setup hint for Task 3: Lower symmetric non-unit MV
!*******************************************************************************

    descrA % TYPE = SPARSE_MATRIX_TYPE_SYMMETRIC
    descrA % MODE = SPARSE_FILL_MODE_UPPER
    descrA % DIAG = SPARSE_DIAG_NON_UNIT

    info = MKL_SPARSE_SET_MV_HINT(cscA, SPARSE_OPERATION_NON_TRANSPOSE, descrA, 1)
    IF ( (info.NE.SPARSE_STATUS_SUCCESS).AND.(info.NE.SPARSE_STATUS_NOT_SUPPORTED) ) THEN
        print *, '  Task 3 Hint: MKL_SPARSE_SET_MV_HINT: '
        print 106, info
        exit_status = 1
        GOTO 99
    END IF

!*******************************************************************************
!  Setup hint for Task 4: General transpose MV
!*******************************************************************************

    descrA % TYPE = SPARSE_MATRIX_TYPE_GENERAL

    info = MKL_SPARSE_SET_MV_HINT(cscA, SPARSE_OPERATION_TRANSPOSE, descrA, 1)
    IF ( (info.NE.SPARSE_STATUS_SUCCESS).AND.(info.NE.SPARSE_STATUS_NOT_SUPPORTED) ) THEN
        print *, '  Task 4 Hint: MKL_SPARSE_SET_MV_HINT: '
        print 106, info
        exit_status = 1
        GOTO 99
    END IF


!*******************************************************************************
!  Analyze hints and optimize:
!*******************************************************************************

    info = MKL_SPARSE_OPTIMIZE(cscA)
    IF (info.NE.SPARSE_STATUS_SUCCESS) THEN
        print *, ' MKL_SPARSE_OPTIMIZE: '
        print 106, info
        exit_status = 1
        GOTO 99
    END IF


!*******************************************************************************
! Task 1: Perform matrix-matrix multiply (L+D)'*sol --> rhs
!    and solve triangular system (L+D)'*temp = rhs with multiple right hand 
!    sides. Array temp must be equal to the array sol.
!*******************************************************************************
    print*
    print*, '  Task 1:                             '
    print*, '     INPUT DATA FOR MKL_SPARSE_D_MM   '
    print*, '     WITH LOWER TRIANGULAR CSC MATRIX '
    print*, '     USING SPARSE_OPERATION_TRANSPOSE '
    print 101, m, nrhs
    print 102, alpha, beta
    print*, ' Input matrix '
    print 104, ((sol(i,j),j=1,nrhs),i=1,m)

    descrA % TYPE = SPARSE_MATRIX_TYPE_TRIANGULAR
    descrA % MODE = SPARSE_FILL_MODE_LOWER
    descrA % DIAG = SPARSE_DIAG_NON_UNIT
    ldx = m
    ldy = m
    info = MKL_SPARSE_D_MM ( SPARSE_OPERATION_TRANSPOSE, alpha, cscA, descrA, SPARSE_LAYOUT_COLUMN_MAJOR, &
                             sol, nrhs, ldx, beta, rhs, ldy )
    IF (info.NE.SPARSE_STATUS_SUCCESS) THEN
        print *, '  Task 1: MKL_SPARSE_D_MM: '
        print 106, info
        exit_status = 1
        GOTO 99
    END IF


    print*
    print*, '     OUTPUT DATA FOR MKL_SPARSE_D_MM  '
    print*, '     WITH LOWER TRIANGULAR CSC MATRIX '
    print 104, ((rhs(i,j),j=1,nrhs),i=1,m)
    print 100
    print*, ' Solve triangular system with obtained '
    print*, ' right hand side                       '

    info = MKL_SPARSE_D_TRSM ( SPARSE_OPERATION_TRANSPOSE, alpha, cscA, descrA, SPARSE_LAYOUT_COLUMN_MAJOR, &
                               rhs, nrhs, ldx, temp, ldy)
    IF (info.NE.SPARSE_STATUS_SUCCESS) THEN
        print *, '  Task 1: MKL_SPARSE_D_TRSM: '
        print 106, info
        exit_status = 1
        GOTO 99
    END IF

    print*
    print*, '     OUTPUT DATA FOR MKL_SPARSE_D_TRSM '
    print 104, ((temp(i,j),j=1,nrhs),i=1,m)

    print*
    print*, 'Validating output data against input data'
    DO j=1,nrhs
        DO i=1,m
            IF (.NOT. ALMOST_EQUAL(sol(i,j), temp(i,j))) THEN
                print*, 'Error in Task 1: output data is not equal to input data'
                exit_status = 1
                GOTO 99
            END IF
        END DO
    END DO
    print*, 'Done'
    print 100

!*******************************************************************************
! Task 2: Perform matrix-vector multiply (U+I)*sol --> rhs
!    and solve triangular system (U+I)*temp = rhs with single right hand 
!    sides. Array temp must be equal to the array sol.
!*******************************************************************************

    print*
    print*, '  Task 2:                                 '
    print*, '     INPUT DATA FOR MKL_SPARSE_D_MV       '
    print*, '     WITH UPPER TRIANGULAR CSC MATRIX     '
    print*, '     USING SPARSE_OPERATION_NON_TRANSPOSE '
    print 102, alpha, beta
    print*, ' Input vector '
    print 105, (sol(i,1),i=1,m)

    descrA % TYPE = SPARSE_MATRIX_TYPE_TRIANGULAR
    descrA % MODE = SPARSE_FILL_MODE_UPPER
    descrA % DIAG = SPARSE_DIAG_UNIT

    info = MKL_SPARSE_D_MV ( SPARSE_OPERATION_NON_TRANSPOSE, alpha, cscA, descrA, sol, beta, rhs)
    IF (info.NE.SPARSE_STATUS_SUCCESS) THEN
        print *, '  Task 2: MKL_SPARSE_D_MV: '
        print 106, info
        exit_status = 1
        GOTO 99
    END IF

    print*
    print*, '     OUTPUT DATA FOR MKL_SPARSE_D_MV  '
    print*, '     WITH UPPER TRIANGULAR CSC MATRIX '
    print 105, (rhs(i,1),i=1,m)
    print 100
    print*, ' Solve triangular system with obtained '
    print*, ' right hand side  '

    info = MKL_SPARSE_D_TRSV ( SPARSE_OPERATION_NON_TRANSPOSE, alpha, cscA, descrA, rhs, temp)
    IF (info.NE.SPARSE_STATUS_SUCCESS) THEN
        print *, '  Task 2: MKL_SPARSE_D_TRSV: '
        print 106, info
        exit_status = 1
        GOTO 99
    END IF

    print*
    print*, '     OUTPUT DATA FOR MKL_SPARSE_D_TRSV '
    print*, '     WITH UPPER TRIANGULAR CSC MATRIX  '
    print 105, (temp(i,1),i=1,m)

    print*
    print*, 'Validating output data against input data'
    DO i=1,m
        IF (.NOT. ALMOST_EQUAL(sol(i,1), temp(i,1))) THEN
            print*, 'Error in Task 2: output data is not equal to input data'
            exit_status = 1
            GOTO 99
        END IF
    END DO
    print*, 'Done'
    print 100

!*******************************************************************************
! Task 3: Perform matrix-vector multiply (L+D+L')*sol --> rhs with the help of
!    MKL_SPARSE_D_MV
!*******************************************************************************
    print*
    print*, '  Task 3:                                 '
    print*, '     INPUT DATA FOR MKL_SPARSE_D_MV       '
    print*, '     WITH LOWER SYMMETRIC CSC MATRIX      '
    print*, '     USING SPARSE_OPERATION_NON_TRANSPOSE '
    print 102, alpha, beta
    print*, ' Input vector '
    print 105, (sol(i, 1),i=1,m)

    descrA % TYPE = SPARSE_MATRIX_TYPE_SYMMETRIC
    descrA % MODE = SPARSE_FILL_MODE_LOWER
    descrA % DIAG = SPARSE_DIAG_NON_UNIT

    info = MKL_SPARSE_D_MV ( SPARSE_OPERATION_NON_TRANSPOSE, alpha, cscA, descrA, sol, beta, rhs)
    IF (info.NE.SPARSE_STATUS_SUCCESS) THEN
        print *, '  Task 3: MKL_SPARSE_D_MV: '
        print 106, info
        exit_status = 1
        GOTO 99
    END IF

    print*
    print*, '     OUTPUT DATA FOR MKL_SPARSE_D_MV '
    print*, '     WITH LOWER SYMMETRIC CSC MATRIX '
    print 105, (rhs(i,1),i=1,m)
    print 100
!*******************************************************************************
! Task 4: Perform matrix-vector multiply alpha*A'*sol + beta*rhs --> rhs with 
!     the help of MKL_SPARSE_D_MV
!*******************************************************************************

    alpha = 2.0
    beta = 1.0

    print*
    print*, '  Task 4:                             '
    print*, '     INPUT DATA FOR MKL_SPARSE_D_MV   '
    print*, '     WITH GENERAL CSC MATRIX          '
    print*, '     USING SPARSE_OPERATION_TRANSPOSE '
    print 102, alpha, beta
    print*, ' Input vector '
    print 105, (sol(i, 1),i=1,m)

    descrA % TYPE = SPARSE_MATRIX_TYPE_GENERAL

    info = MKL_SPARSE_D_MV ( SPARSE_OPERATION_TRANSPOSE, alpha, cscA, descrA, sol, beta, rhs)
    IF (info.NE.SPARSE_STATUS_SUCCESS) THEN
        print *, '  Task 4: MKL_SPARSE_D_MV: '
        print 106, info
        exit_status = 1
        GOTO 99
    END IF

    write(*,*) ' info = ', info

    print*
    print*, '     OUTPUT DATA FOR MKL_SPARSE_D_MV '
    print*, '     WITH GENERIC CSC MATRIX '
    print 105, (rhs(i,1),i=1,m)
    print 100
   
 100      format('------------------------------------------------')
 101      format(7x,'M=',i1,'  NRHS=',i1)
 102      format(7x,'ALPHA = ',f4.1,' BETA = ', f4.1)
 103      format(7x,'TRANS = ',a1)
 104      format(2(f7.1, 3x))
 105      format(f4.1)
 106      format(7x,'ERROR, INFO=',i1)

 99 CONTINUE

 !   Release internal representation of CSC matrix
    info = MKL_SPARSE_DESTROY(cscA)
    IF (info.NE.SPARSE_STATUS_SUCCESS) THEN
        print *, '  MKL_SPARSE_DESTROY: '
        print 106, info
        exit_status = 1
    END IF

    call exit(exit_status)

CONTAINS
  LOGICAL FUNCTION ALMOST_EQUAL(a, b)
    IMPLICIT NONE
    DOUBLE PRECISION, INTENT(IN) :: a, b
    ALMOST_EQUAL = .FALSE.
    IF (ABS(a-b) .LT. 1e-10) THEN
       ALMOST_EQUAL = .TRUE.
    END IF
    RETURN
  END FUNCTION ALMOST_EQUAL
END PROGRAM SPARSE_D_CSR_EXAMPLE
