#
# Copyright (C) 2020 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#


# ------------------------------------------------------------------------------
# This example shows how to obtain GPU roofs.
# For example, the expected output might look like:
#
# DP Vector FMA Peak 108 GFLOPS
# DP Vector Add Peak 55 GFLOPS
# SP Vector FMA Peak 439 GFLOPS
# SP Vector Add Peak 220 GFLOPS
# Int64 Vector Add Peak 55 GFLOPS
# Int32 Vector Add Peak 220 GFLOPS
# Int16 Vector Add Peak 438 GFLOPS
# Int8 Vector Add Peak 217 GFLOPS
# SLM Bandwidth 203 GB/s
# L3 Bandwidth 203 GB/s
# DRAM Bandwidth 29 GB/s
# GTI Bandwidth 73 GB/s
# ------------------------------------------------------------------------------

import math
import sys

try:

    import advisor

except ImportError:

    print(
        """Import error: Python could not resolve path to Advisor's pythonapi directory.
        To fix, either manually add path to the pythonapi directory into PYTHONPATH environment
        variable, or use advixe-vars.* scripts to set up product environment variables automatically."""
    )
    sys.exit(1)

# Check command-line arguments.
if len(sys.argv) < 2:
    print('Usage: "python {} path_to_project_dir"'.format(__file__))
    sys.exit(2)

# Open the Advisor Project and load the data.
project = advisor.open_project(sys.argv[1])
data = project.load(advisor.SURVEY)

# The data.get_gpu_roofs() returns GPU roofs. It is equvalient to data.gpu_roofs.
roofs = data.get_gpu_roofs()

for roof in roofs:

    # Process memory roofs.
    if roof.has_type(advisor.RoofType.MEMORY):

        # Convert to GByte/s.
        bandwidth = roof.bandwidth // math.pow(10, 9)
        print("{} {:.0f} GB/s".format(roof.name, bandwidth))

    # Process compute roofs.
    else:

        # Convert to GFLOPS.
        bandwidth = roof.bandwidth // math.pow(10, 9)
        print("{} {:.0f} GFLOPS".format(roof.name, bandwidth))
