#
# Copyright (C) 2015 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#

import sys
import collections
import inspect

if sys.version_info[0] == 3:
    def cast_lnotab_char(ch):
        return ch
    def cast_to_bytes(s):
        return bytes(s, 'ascii', 'ignore')
else:
    cast_lnotab_char = ord
    def cast_to_bytes(s):
        return s

def getCodeRange(code):
    return code.co_firstlineno, \
           code.co_firstlineno + sum([cast_lnotab_char(ch) for ch in code.co_lnotab[1::2]])

# constants from "opcode.h"
STORE_LOAD_INSTR_LEN = 3
BUILD_CLASS_OPCODE_LEN = 1

def getNestedCodeString(code, subcode):
    codeStart, codeStop = None, None
    if isinstance(subcode, int):
        subcode = code.co_consts[subcode]
    subStart, subStop = getCodeRange(subcode)
    codePos, linePos = 0, code.co_firstlineno
    ltab = code.co_lnotab
    for posDiff, lineDiff in zip(*[iter(ltab)]*2):
        codePos += cast_lnotab_char(posDiff)
        linePos += cast_lnotab_char(lineDiff)
        if codeStart is None:
            if linePos >= subStart:
                codeStart = codePos
        elif linePos >= subStop:
            codeStop = codePos
            break
    if codeStart is None:
        codeStart = 0
        #raise ValueError('Cannot find subcode generating code')
    if codeStop is None:
        codeStop = codePos
    return code.co_code[codeStart:codeStop - STORE_LOAD_INSTR_LEN]

def funcWithNestedClass():
    class C:
        pass
    pass
BUILD_CLASS_OPCODE = getNestedCodeString(funcWithNestedClass.__code__,
                                     [c for c in funcWithNestedClass.__code__.co_consts if inspect.iscode(c)][0]) \
                         [-BUILD_CLASS_OPCODE_LEN:]

def isNestedClass(code, subcode):
    return getNestedCodeString(code, subcode).endswith(BUILD_CLASS_OPCODE)


class CodeObjectType:
    Module, Function, Method, Class, Unknown = range(5)

CodePathElement = collections.namedtuple('CodePathElement', 'type name')
def getCodeChildren(codeObject, codeType = CodeObjectType.Unknown, parentRoot = None):
    if not parentRoot:
        parentRoot = []
    rootPath = parentRoot + [CodePathElement(codeType, codeObject.co_name)]
    result = [(rootPath, codeObject)]
    for obj in codeObject.co_consts:
        # walking nested code objects
        if inspect.iscode(obj):
            subType = CodeObjectType.Class if isNestedClass(codeObject, obj) else CodeObjectType.Function
            result.extend(getCodeChildren(obj, subType, rootPath))
    return result

def formatargspec(args, varargs, varkw):
    specs = args.copy()
    if varargs:
        specs.append('*' + varargs)
    if varkw:
        specs.append('**' + varkw)
    return '(' + ', '.join(specs) + ')'

def getCodeArgsStr(codeType, code):
    if codeType in (CodeObjectType.Function, CodeObjectType.Method):
        args, varargs, varkw = inspect.getargs(code)
        return formatargspec(args, varargs, varkw)
    else:
        return ''

def getModuleCallables(fileName, asDict = True):
    with open(fileName, 'rb') as input:
        # * adding ending newline to make compile() happy on pre-2.7 Python
        # * adding "dummydummy = None" so that module won't end with class declaration
        #      because if it does getNestedCodeString() isn't able to extract code string
        #      that builds the declaration as it relies on line numbers to find
        #      the position in the code string, and if module has no body its last line
        #      in code object would be the one declaring class
        code = compile(input.read() + cast_to_bytes('\n#dummy code\ndummydummy = None\n'), fileName, 'exec')
    analyzed = getCodeChildren(code, CodeObjectType.Module)
    if asDict:
        result = collections.defaultdict(list)
        for codePath, codeObj in analyzed:
            result[codeObj.co_firstlineno].append(getCodeRange(codeObj) + (codePath,
                                                   getCodeArgsStr(codePath[-1].type, codeObj),
                                                   hash(codeObj)))
        return dict(result)
    else:
        return [(getCodeRange(codeObj) + (codePath, getCodeArgsStr(codePath[-1].type, codeObj), hash(codeObj))) for (codePath, codeObj) in analyzed]

if __name__ == '__main__':
    if len(sys.argv) < 2:
        print('Usage: %s <fileName>' % sys.argv[0])
        sys.exit(1)

    testLambdas = (lambda: 1, lambda bb: 2, lambda *args: 3)

    allCallables = getModuleCallables(sys.argv[1])
    for line  in sorted(allCallables.keys()):
        print('%d:' % line)
        for (codeStart, codeStop, codePath, codeArgs, codeHash) in allCallables[line]:
            codePathView = []
            for (codeType, codeName) in codePath:
                if codeType == CodeObjectType.Class:
                    codePathView.extend([codeName, '.'])
                elif codeType == CodeObjectType.Function:
                    codePathView.extend([codeName, '::'])
                elif codeType == CodeObjectType.Module:
                    codePathView.extend([codeName, '.'])
            print('    %s%s  [%X] %d-%d' % (''.join(codePathView[:-1]), codeArgs, codeHash, codeStart, codeStop))
