﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/managedblockchain/ManagedBlockchain_EXPORTS.h>
#include <aws/managedblockchain/model/NodeFrameworkAttributes.h>
#include <aws/managedblockchain/model/NodeLogPublishingConfiguration.h>
#include <aws/managedblockchain/model/NodeStatus.h>
#include <aws/managedblockchain/model/StateDBType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace ManagedBlockchain {
namespace Model {

/**
 * <p>Configuration properties of a node.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/Node">AWS
 * API Reference</a></p>
 */
class Node {
 public:
  AWS_MANAGEDBLOCKCHAIN_API Node() = default;
  AWS_MANAGEDBLOCKCHAIN_API Node(Aws::Utils::Json::JsonView jsonValue);
  AWS_MANAGEDBLOCKCHAIN_API Node& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_MANAGEDBLOCKCHAIN_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The unique identifier of the network that the node is on.</p>
   */
  inline const Aws::String& GetNetworkId() const { return m_networkId; }
  inline bool NetworkIdHasBeenSet() const { return m_networkIdHasBeenSet; }
  template <typename NetworkIdT = Aws::String>
  void SetNetworkId(NetworkIdT&& value) {
    m_networkIdHasBeenSet = true;
    m_networkId = std::forward<NetworkIdT>(value);
  }
  template <typename NetworkIdT = Aws::String>
  Node& WithNetworkId(NetworkIdT&& value) {
    SetNetworkId(std::forward<NetworkIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the member to which the node belongs.</p> <p>Applies
   * only to Hyperledger Fabric.</p>
   */
  inline const Aws::String& GetMemberId() const { return m_memberId; }
  inline bool MemberIdHasBeenSet() const { return m_memberIdHasBeenSet; }
  template <typename MemberIdT = Aws::String>
  void SetMemberId(MemberIdT&& value) {
    m_memberIdHasBeenSet = true;
    m_memberId = std::forward<MemberIdT>(value);
  }
  template <typename MemberIdT = Aws::String>
  Node& WithMemberId(MemberIdT&& value) {
    SetMemberId(std::forward<MemberIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the node.</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  Node& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The instance type of the node.</p>
   */
  inline const Aws::String& GetInstanceType() const { return m_instanceType; }
  inline bool InstanceTypeHasBeenSet() const { return m_instanceTypeHasBeenSet; }
  template <typename InstanceTypeT = Aws::String>
  void SetInstanceType(InstanceTypeT&& value) {
    m_instanceTypeHasBeenSet = true;
    m_instanceType = std::forward<InstanceTypeT>(value);
  }
  template <typename InstanceTypeT = Aws::String>
  Node& WithInstanceType(InstanceTypeT&& value) {
    SetInstanceType(std::forward<InstanceTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Availability Zone in which the node exists. Required for Ethereum nodes.
   * </p>
   */
  inline const Aws::String& GetAvailabilityZone() const { return m_availabilityZone; }
  inline bool AvailabilityZoneHasBeenSet() const { return m_availabilityZoneHasBeenSet; }
  template <typename AvailabilityZoneT = Aws::String>
  void SetAvailabilityZone(AvailabilityZoneT&& value) {
    m_availabilityZoneHasBeenSet = true;
    m_availabilityZone = std::forward<AvailabilityZoneT>(value);
  }
  template <typename AvailabilityZoneT = Aws::String>
  Node& WithAvailabilityZone(AvailabilityZoneT&& value) {
    SetAvailabilityZone(std::forward<AvailabilityZoneT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Attributes of the blockchain framework being used.</p>
   */
  inline const NodeFrameworkAttributes& GetFrameworkAttributes() const { return m_frameworkAttributes; }
  inline bool FrameworkAttributesHasBeenSet() const { return m_frameworkAttributesHasBeenSet; }
  template <typename FrameworkAttributesT = NodeFrameworkAttributes>
  void SetFrameworkAttributes(FrameworkAttributesT&& value) {
    m_frameworkAttributesHasBeenSet = true;
    m_frameworkAttributes = std::forward<FrameworkAttributesT>(value);
  }
  template <typename FrameworkAttributesT = NodeFrameworkAttributes>
  Node& WithFrameworkAttributes(FrameworkAttributesT&& value) {
    SetFrameworkAttributes(std::forward<FrameworkAttributesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Configuration properties for logging events associated with a peer node on a
   * Hyperledger Fabric network on Managed Blockchain.</p>
   */
  inline const NodeLogPublishingConfiguration& GetLogPublishingConfiguration() const { return m_logPublishingConfiguration; }
  inline bool LogPublishingConfigurationHasBeenSet() const { return m_logPublishingConfigurationHasBeenSet; }
  template <typename LogPublishingConfigurationT = NodeLogPublishingConfiguration>
  void SetLogPublishingConfiguration(LogPublishingConfigurationT&& value) {
    m_logPublishingConfigurationHasBeenSet = true;
    m_logPublishingConfiguration = std::forward<LogPublishingConfigurationT>(value);
  }
  template <typename LogPublishingConfigurationT = NodeLogPublishingConfiguration>
  Node& WithLogPublishingConfiguration(LogPublishingConfigurationT&& value) {
    SetLogPublishingConfiguration(std::forward<LogPublishingConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The state database that the node uses. Values are <code>LevelDB</code> or
   * <code>CouchDB</code>.</p> <p>Applies only to Hyperledger Fabric.</p>
   */
  inline StateDBType GetStateDB() const { return m_stateDB; }
  inline bool StateDBHasBeenSet() const { return m_stateDBHasBeenSet; }
  inline void SetStateDB(StateDBType value) {
    m_stateDBHasBeenSet = true;
    m_stateDB = value;
  }
  inline Node& WithStateDB(StateDBType value) {
    SetStateDB(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the node.</p> <ul> <li> <p> <code>CREATING</code> - The Amazon
   * Web Services account is in the process of creating a node.</p> </li> <li> <p>
   * <code>AVAILABLE</code> - The node has been created and can participate in the
   * network.</p> </li> <li> <p> <code>UNHEALTHY</code> - The node is impaired and
   * might not function as expected. Amazon Managed Blockchain automatically finds
   * nodes in this state and tries to recover them. If a node is recoverable, it
   * returns to <code>AVAILABLE</code>. Otherwise, it moves to <code>FAILED</code>
   * status.</p> </li> <li> <p> <code>CREATE_FAILED</code> - The Amazon Web Services
   * account attempted to create a node and creation failed.</p> </li> <li> <p>
   * <code>UPDATING</code> - The node is in the process of being updated.</p> </li>
   * <li> <p> <code>DELETING</code> - The node is in the process of being
   * deleted.</p> </li> <li> <p> <code>DELETED</code> - The node can no longer
   * participate on the network.</p> </li> <li> <p> <code>FAILED</code> - The node is
   * no longer functional, cannot be recovered, and must be deleted.</p> </li> <li>
   * <p> <code>INACCESSIBLE_ENCRYPTION_KEY</code> - The node is impaired and might
   * not function as expected because it cannot access the specified customer managed
   * key in KMS for encryption at rest. Either the KMS key was disabled or deleted,
   * or the grants on the key were revoked.</p> <p>The effect of disabling or
   * deleting a key or of revoking a grant isn't immediate. It might take some time
   * for the node resource to discover that the key is inaccessible. When a resource
   * is in this state, we recommend deleting and recreating the resource.</p> </li>
   * </ul>
   */
  inline NodeStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(NodeStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline Node& WithStatus(NodeStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time that the node was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationDate() const { return m_creationDate; }
  inline bool CreationDateHasBeenSet() const { return m_creationDateHasBeenSet; }
  template <typename CreationDateT = Aws::Utils::DateTime>
  void SetCreationDate(CreationDateT&& value) {
    m_creationDateHasBeenSet = true;
    m_creationDate = std::forward<CreationDateT>(value);
  }
  template <typename CreationDateT = Aws::Utils::DateTime>
  Node& WithCreationDate(CreationDateT&& value) {
    SetCreationDate(std::forward<CreationDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags assigned to the node. Each tag consists of a key and optional value.</p>
   * <p>For more information about tags, see <a
   * href="https://docs.aws.amazon.com/managed-blockchain/latest/ethereum-dev/tagging-resources.html">Tagging
   * Resources</a> in the <i>Amazon Managed Blockchain Ethereum Developer Guide</i>,
   * or <a
   * href="https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html">Tagging
   * Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer
   * Guide</i>.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  Node& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  Node& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the node. For more information about ARNs
   * and their format, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Names (ARNs)</a> in the <i>Amazon Web Services General
   * Reference</i>.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  Node& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the customer managed key in Key Management
   * Service (KMS) that the node uses for encryption at rest. If the value of this
   * parameter is <code>"AWS Owned KMS Key"</code>, the node uses an Amazon Web
   * Services owned KMS key for encryption. The node inherits this parameter from the
   * member that it belongs to.</p> <p>For more information, see <a
   * href="https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/managed-blockchain-encryption-at-rest.html">Encryption
   * at Rest</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer
   * Guide</i>.</p> <p>Applies only to Hyperledger Fabric.</p>
   */
  inline const Aws::String& GetKmsKeyArn() const { return m_kmsKeyArn; }
  inline bool KmsKeyArnHasBeenSet() const { return m_kmsKeyArnHasBeenSet; }
  template <typename KmsKeyArnT = Aws::String>
  void SetKmsKeyArn(KmsKeyArnT&& value) {
    m_kmsKeyArnHasBeenSet = true;
    m_kmsKeyArn = std::forward<KmsKeyArnT>(value);
  }
  template <typename KmsKeyArnT = Aws::String>
  Node& WithKmsKeyArn(KmsKeyArnT&& value) {
    SetKmsKeyArn(std::forward<KmsKeyArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_networkId;

  Aws::String m_memberId;

  Aws::String m_id;

  Aws::String m_instanceType;

  Aws::String m_availabilityZone;

  NodeFrameworkAttributes m_frameworkAttributes;

  NodeLogPublishingConfiguration m_logPublishingConfiguration;

  StateDBType m_stateDB{StateDBType::NOT_SET};

  NodeStatus m_status{NodeStatus::NOT_SET};

  Aws::Utils::DateTime m_creationDate{};

  Aws::Map<Aws::String, Aws::String> m_tags;

  Aws::String m_arn;

  Aws::String m_kmsKeyArn;
  bool m_networkIdHasBeenSet = false;
  bool m_memberIdHasBeenSet = false;
  bool m_idHasBeenSet = false;
  bool m_instanceTypeHasBeenSet = false;
  bool m_availabilityZoneHasBeenSet = false;
  bool m_frameworkAttributesHasBeenSet = false;
  bool m_logPublishingConfigurationHasBeenSet = false;
  bool m_stateDBHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_creationDateHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_arnHasBeenSet = false;
  bool m_kmsKeyArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace ManagedBlockchain
}  // namespace Aws
