﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/pinpoint-sms-voice-v2/PinpointSMSVoiceV2Request.h>
#include <aws/pinpoint-sms-voice-v2/PinpointSMSVoiceV2_EXPORTS.h>
#include <aws/pinpoint-sms-voice-v2/model/MessageType.h>

#include <utility>

namespace Aws {
namespace PinpointSMSVoiceV2 {
namespace Model {

/**
 */
class SetDefaultMessageTypeRequest : public PinpointSMSVoiceV2Request {
 public:
  AWS_PINPOINTSMSVOICEV2_API SetDefaultMessageTypeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "SetDefaultMessageType"; }

  AWS_PINPOINTSMSVOICEV2_API Aws::String SerializePayload() const override;

  AWS_PINPOINTSMSVOICEV2_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The configuration set to update with a new default message type. This field
   * can be the ConsigurationSetName or ConfigurationSetArn.</p>
   */
  inline const Aws::String& GetConfigurationSetName() const { return m_configurationSetName; }
  inline bool ConfigurationSetNameHasBeenSet() const { return m_configurationSetNameHasBeenSet; }
  template <typename ConfigurationSetNameT = Aws::String>
  void SetConfigurationSetName(ConfigurationSetNameT&& value) {
    m_configurationSetNameHasBeenSet = true;
    m_configurationSetName = std::forward<ConfigurationSetNameT>(value);
  }
  template <typename ConfigurationSetNameT = Aws::String>
  SetDefaultMessageTypeRequest& WithConfigurationSetName(ConfigurationSetNameT&& value) {
    SetConfigurationSetName(std::forward<ConfigurationSetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of message. Valid values are TRANSACTIONAL for messages that are
   * critical or time-sensitive and PROMOTIONAL for messages that aren't critical or
   * time-sensitive.</p>
   */
  inline MessageType GetMessageType() const { return m_messageType; }
  inline bool MessageTypeHasBeenSet() const { return m_messageTypeHasBeenSet; }
  inline void SetMessageType(MessageType value) {
    m_messageTypeHasBeenSet = true;
    m_messageType = value;
  }
  inline SetDefaultMessageTypeRequest& WithMessageType(MessageType value) {
    SetMessageType(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_configurationSetName;

  MessageType m_messageType{MessageType::NOT_SET};
  bool m_configurationSetNameHasBeenSet = false;
  bool m_messageTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace PinpointSMSVoiceV2
}  // namespace Aws
