﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/simspaceweaver/SimSpaceWeaverRequest.h>
#include <aws/simspaceweaver/SimSpaceWeaver_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace SimSpaceWeaver {
namespace Model {

/**
 */
class ListSimulationsRequest : public SimSpaceWeaverRequest {
 public:
  AWS_SIMSPACEWEAVER_API ListSimulationsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListSimulations"; }

  AWS_SIMSPACEWEAVER_API Aws::String SerializePayload() const override;

  AWS_SIMSPACEWEAVER_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The maximum number of simulations to list.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListSimulationsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If SimSpace Weaver returns <code>nextToken</code>, then there are more
   * results available. The value of <code>nextToken</code> is a unique pagination
   * token for each page. To retrieve the next page, call the operation again using
   * the returned token. Keep all other arguments unchanged. If no results remain,
   * then <code>nextToken</code> is set to <code>null</code>. Each pagination token
   * expires after 24 hours. If you provide a token that isn't valid, then you
   * receive an <i>HTTP 400 ValidationException</i> error.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListSimulationsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace SimSpaceWeaver
}  // namespace Aws
