/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.util.xml;

import ghidra.app.cmd.refs.AddMemRefCmd;
import ghidra.app.cmd.refs.AddOffsetMemRefCmd;
import ghidra.app.cmd.refs.AddStackRefCmd;
import ghidra.app.cmd.refs.SetExternalRefCmd;
import ghidra.app.cmd.refs.SetPrimaryRefCmd;
import ghidra.app.util.importer.MessageLog;
import ghidra.framework.model.DomainObject;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressFactory;
import ghidra.program.model.address.AddressFormatException;
import ghidra.program.model.address.AddressIterator;
import ghidra.program.model.address.AddressSetView;
import ghidra.program.model.listing.CodeUnit;
import ghidra.program.model.listing.Instruction;
import ghidra.program.model.listing.Listing;
import ghidra.program.model.listing.Program;
import ghidra.program.model.scalar.Scalar;
import ghidra.program.model.symbol.Equate;
import ghidra.program.model.symbol.EquateReference;
import ghidra.program.model.symbol.EquateTable;
import ghidra.program.model.symbol.ExternalLocation;
import ghidra.program.model.symbol.ExternalReference;
import ghidra.program.model.symbol.FlowType;
import ghidra.program.model.symbol.OffsetReference;
import ghidra.program.model.symbol.RefType;
import ghidra.program.model.symbol.Reference;
import ghidra.program.model.symbol.ReferenceManager;
import ghidra.program.model.symbol.SourceType;
import ghidra.program.model.symbol.StackReference;
import ghidra.util.Msg;
import ghidra.util.XmlProgramUtilities;
import ghidra.util.exception.AssertException;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.DuplicateNameException;
import ghidra.util.exception.InvalidInputException;
import ghidra.util.task.TaskMonitor;
import ghidra.util.xml.XmlAttributeException;
import ghidra.util.xml.XmlAttributes;
import ghidra.util.xml.XmlUtilities;
import ghidra.util.xml.XmlWriter;
import ghidra.xml.XmlElement;
import ghidra.xml.XmlPullParser;
import java.util.Iterator;
import java.util.LinkedList;
import org.xml.sax.SAXParseException;

class MarkupXmlMgr {
    private Program program;
    private ReferenceManager refManager;
    private AddressFactory factory;
    private MessageLog log;
    private EquateTable equateTable;
    private Listing listing;

    MarkupXmlMgr(Program program, MessageLog log) {
        this.program = program;
        this.listing = program.getListing();
        this.refManager = program.getReferenceManager();
        this.equateTable = program.getEquateTable();
        this.factory = program.getAddressFactory();
        this.log = log;
    }

    void read(XmlPullParser parser, boolean overwrite, boolean isExtLibs, boolean isFunctions, boolean ignoreStackReferences, TaskMonitor monitor) throws SAXParseException, CancelledException {
        XmlElement element = parser.next();
        if (!element.isStart() || !element.getName().equals("MARKUP")) {
            throw new SAXParseException("Expected MARKUP start tag", null, null, parser.getLineNumber(), parser.getColumnNumber());
        }
        element = parser.next();
        while (element.isStart()) {
            if (monitor.isCancelled()) {
                throw new CancelledException();
            }
            String tagName = element.getName().toUpperCase();
            if (tagName.equals("MEMORY_REFERENCE")) {
                this.processMemoryReference(element, overwrite);
            } else if (tagName.equals("STACK_REFERENCE")) {
                if (isFunctions && !ignoreStackReferences) {
                    this.processStackReference(element, overwrite);
                }
            } else if (tagName.equals("EXT_LIBRARY_REFERENCE")) {
                if (isExtLibs) {
                    this.processExtLibraryReference(element, overwrite);
                }
            } else if (tagName.equals("EQUATE_REFERENCE")) {
                this.processEquateReference(element, overwrite);
            } else if (!tagName.equals("MANUAL_OPERAND") && !tagName.equals("MANUAL_INSTRUCTION")) {
                throw new SAXParseException("Unexpected XML tag: " + tagName, null, null, parser.getLineNumber(), parser.getColumnNumber());
            }
            element = parser.next();
            if (element.isStart() || !element.getName().equalsIgnoreCase(tagName)) {
                throw new SAXParseException("Expected " + tagName + " end tag", null, null, parser.getLineNumber(), parser.getColumnNumber());
            }
            element = parser.next();
        }
        if (!element.getName().equals("MARKUP")) {
            throw new SAXParseException("Expected MARKUP end tag", null, null, parser.getLineNumber(), parser.getColumnNumber());
        }
    }

    private RefType getDefaultRefType(Address fromAddr, Address toAddr, int opIndex) {
        CodeUnit srcCU = this.program.getListing().getCodeUnitAt(fromAddr);
        CodeUnit destCU = this.program.getListing().getCodeUnitAt(toAddr);
        if (srcCU instanceof Instruction && destCU instanceof Instruction) {
            Instruction srcInst = (Instruction)srcCU;
            FlowType ft = srcInst.getFlowType();
            if (ft.isCall() || ft.isJump()) {
                return ft;
            }
        } else if (srcCU instanceof Instruction && opIndex != -1) {
            Instruction srcInst = (Instruction)srcCU;
            return srcInst.getOperandRefType(opIndex);
        }
        return RefType.DATA;
    }

    private void processMemoryReference(XmlElement element, boolean overwrite) {
        try {
            Reference existingMemRef;
            String fromAddrStr = element.getAttribute("ADDRESS");
            if (fromAddrStr == null) {
                throw new XmlAttributeException("ADDRESS attribute missing for MEMORY_REFERENCE element");
            }
            Address fromAddr = XmlProgramUtilities.parseAddress((AddressFactory)this.factory, (String)fromAddrStr);
            if (fromAddr == null) {
                throw new AddressFormatException("Incompatible Memory Reference FROM Address: " + fromAddrStr);
            }
            String toAddrStr = element.getAttribute("TO_ADDRESS");
            if (toAddrStr == null) {
                throw new XmlAttributeException("TO_ADDRESS attribute missing for MEMORY_REFERENCE element");
            }
            Address toAddr = XmlProgramUtilities.parseAddress((AddressFactory)this.factory, (String)toAddrStr);
            if (toAddr == null) {
                throw new AddressFormatException("Incompatible Memory Reference TO Address: " + toAddrStr);
            }
            int opIndex = -1;
            if (element.hasAttribute("OPERAND_INDEX") && (opIndex = XmlUtilities.parseInt((String)element.getAttribute("OPERAND_INDEX"))) < 0) {
                throw new XmlAttributeException("Illegal OPERAND_INDEX value [" + opIndex + "]");
            }
            boolean userDefined = true;
            if (element.hasAttribute("USER_DEFINED")) {
                userDefined = XmlUtilities.parseBoolean((String)element.getAttribute("USER_DEFINED"));
            }
            boolean primary = false;
            if (element.hasAttribute("PRIMARY")) {
                primary = XmlUtilities.parseBoolean((String)element.getAttribute("PRIMARY"));
            }
            Address baseAddr = null;
            if (element.hasAttribute("BASE_ADDRESS")) {
                baseAddr = XmlProgramUtilities.parseAddress((AddressFactory)this.factory, (String)element.getAttribute("BASE_ADDRESS"));
            }
            if (!overwrite && (existingMemRef = this.refManager.getReference(fromAddr, toAddr, opIndex)) != null) {
                this.log.appendMsg("Memory reference already existed from [" + String.valueOf(fromAddr) + "] to [" + String.valueOf(toAddr) + "] on operand [" + opIndex + "]");
                return;
            }
            RefType refType = this.getDefaultRefType(fromAddr, toAddr, opIndex);
            Object cmd = null;
            if (baseAddr != null) {
                long offset = toAddr.subtract(baseAddr);
                cmd = new AddOffsetMemRefCmd(fromAddr, toAddr, false, refType, userDefined ? SourceType.USER_DEFINED : SourceType.DEFAULT, opIndex, offset);
            } else {
                cmd = new AddMemRefCmd(fromAddr, toAddr, refType, userDefined ? SourceType.USER_DEFINED : SourceType.DEFAULT, opIndex);
            }
            cmd.applyTo((DomainObject)this.program);
            cmd = new SetPrimaryRefCmd(fromAddr, opIndex, toAddr, primary);
            cmd.applyTo((DomainObject)this.program);
        }
        catch (Exception e) {
            this.log.appendException((Throwable)e);
        }
    }

    private void processStackReference(XmlElement element, boolean overwrite) {
        try {
            CodeUnit cu;
            String addrStr = element.getAttribute("ADDRESS");
            if (addrStr == null) {
                throw new XmlAttributeException("ADDRESS attribute missing for STACK_REFERENCE element");
            }
            Address addr = XmlProgramUtilities.parseAddress((AddressFactory)this.factory, (String)addrStr);
            if (addr == null) {
                throw new AddressFormatException("Incompatible Stack Reference Address: " + addrStr);
            }
            int opIndex = -1;
            if (element.hasAttribute("OPERAND_INDEX")) {
                opIndex = XmlUtilities.parseInt((String)element.getAttribute("OPERAND_INDEX"));
            }
            if ((cu = this.listing.getCodeUnitAt(addr)) == null) {
                this.log.appendMsg("No codeunit at " + String.valueOf(addr));
                return;
            }
            Reference ref = cu.getPrimaryReference(opIndex);
            if (ref != null && !overwrite) {
                return;
            }
            boolean userDefined = true;
            if (element.hasAttribute("USER_DEFINED")) {
                userDefined = XmlUtilities.parseBoolean((String)element.getAttribute("USER_DEFINED"));
            }
            int offset = XmlUtilities.parseInt((String)element.getAttribute("STACK_PTR_OFFSET"));
            AddStackRefCmd addCmd = new AddStackRefCmd(addr, opIndex, offset, userDefined ? SourceType.USER_DEFINED : SourceType.DEFAULT);
            addCmd.applyTo(this.program);
        }
        catch (Exception e) {
            this.log.appendException((Throwable)e);
        }
    }

    private void processExtLibraryReference(XmlElement element, boolean overwrite) {
        try {
            Address libAddr;
            String addrStr = element.getAttribute("ADDRESS");
            if (addrStr == null) {
                throw new XmlAttributeException("ADDRESS attribute missing for EXT_LIBRARY_REFERENCE element");
            }
            Address addr = XmlProgramUtilities.parseAddress((AddressFactory)this.factory, (String)addrStr);
            if (addr == null) {
                throw new AddressFormatException("Incompatible External Reference Address: " + addrStr);
            }
            int opIndex = -1;
            if (element.hasAttribute("OPERAND_INDEX")) {
                opIndex = XmlUtilities.parseInt((String)element.getAttribute("OPERAND_INDEX"));
            }
            boolean userDefined = true;
            if (element.hasAttribute("USER_DEFINED")) {
                userDefined = XmlUtilities.parseBoolean((String)element.getAttribute("USER_DEFINED"));
            }
            String programName = element.getAttribute("LIB_PROG_NAME");
            String label = element.getAttribute("LIB_LABEL");
            Address address = libAddr = element.hasAttribute("LIB_ADDR") ? this.factory.getAddress(element.getAttribute("LIB_ADDR")) : null;
            if (label == null && libAddr == null) {
                this.log.appendMsg("External library reference for address " + String.valueOf(addr) + " does not have a label or an external address specified at " + element.getLineNumber() + ". External reference will not be created");
                return;
            }
            CodeUnit cu = this.listing.getCodeUnitAt(addr);
            if (cu == null) {
                this.log.appendMsg("No codeunit at " + String.valueOf(addr));
                return;
            }
            ExternalReference ref = cu.getExternalReference(opIndex);
            if (ref != null) {
                if (!overwrite) {
                    return;
                }
                this.program.getReferenceManager().delete((Reference)ref);
            }
            SetExternalRefCmd addCmd = new SetExternalRefCmd(addr, opIndex, programName, label, libAddr, userDefined ? SourceType.USER_DEFINED : SourceType.IMPORTED);
            addCmd.applyTo(this.program);
        }
        catch (Exception e) {
            this.log.appendException((Throwable)e);
        }
    }

    private void processEquateReference(XmlElement element, boolean overwrite) {
        try {
            Equate existingEquate;
            String addrStr = element.getAttribute("ADDRESS");
            if (addrStr == null) {
                throw new XmlAttributeException("ADDRESS attribute missing for EQUATE_REFERENCE element");
            }
            Address addr = XmlProgramUtilities.parseAddress((AddressFactory)this.factory, (String)addrStr);
            if (addr == null) {
                throw new AddressFormatException("Incompatible Equate Reference Address: " + addrStr);
            }
            if (this.listing.isUndefined(addr, addr)) {
                this.log.appendMsg("BAD EQUATE REFERENCE: defined code unit not found at " + String.valueOf(addr));
                return;
            }
            CodeUnit cu = this.listing.getCodeUnitAt(addr);
            if (cu == null) {
                this.log.appendMsg("No codeunit at " + String.valueOf(addr));
                return;
            }
            String equateName = element.getAttribute("NAME");
            int opIndex = -1;
            LinkedList<Scalar> instrScalars = new LinkedList<Scalar>();
            if (element.hasAttribute("OPERAND_INDEX") && (opIndex = XmlUtilities.parseInt((String)element.getAttribute("OPERAND_INDEX"))) != -1) {
                Scalar tempScalar = cu.getScalar(opIndex);
                if (tempScalar != null) {
                    instrScalars.add(tempScalar);
                } else if (cu instanceof Instruction) {
                    Instruction instr = (Instruction)cu;
                    Object[] opObjects = instr.getOpObjects(opIndex);
                    for (int i = 0; i < opObjects.length; ++i) {
                        if (!(opObjects[i] instanceof Scalar)) continue;
                        instrScalars.add((Scalar)opObjects[i]);
                    }
                    if (instrScalars.size() == 0) {
                        this.log.appendMsg("BAD EQUATE REFERENCE: operand [" + opIndex + "] at address [" + String.valueOf(addr) + "] is not a scalar.");
                        return;
                    }
                }
            }
            long value = 0L;
            if (element.hasAttribute("VALUE")) {
                value = XmlUtilities.parseLong((String)element.getAttribute("VALUE"));
                Scalar matchingScalar = null;
                Iterator itr = instrScalars.iterator();
                while (matchingScalar == null && itr.hasNext()) {
                    matchingScalar = (Scalar)itr.next();
                    if (matchingScalar.getSignedValue() == value) continue;
                    matchingScalar = null;
                }
                if (matchingScalar == null) {
                    this.log.appendMsg("BAD EQUATE REFERENCE: equate [" + equateName + "] value [0x" + Long.toHexString(value) + "] does not match scalar on operand [" + opIndex + "] at address [" + String.valueOf(addr) + "]");
                    return;
                }
            } else if (instrScalars.size() > 0) {
                Msg.warn((Object)this, (Object)"NO VALUE SPECIFIED");
                value = ((Scalar)instrScalars.get(0)).getSignedValue();
            } else {
                this.log.appendMsg("BAD EQUATE REFERENCE: either the VALUE or OPERAND_INDEX must be specified");
                return;
            }
            Equate equate = this.equateTable.getEquate(equateName);
            if (equate != null) {
                if (value != equate.getValue()) {
                    this.log.appendMsg("BAD EQUATE REFERENCE: equate [" + equateName + "] value [0x" + Long.toHexString(value) + "] conflicts with existing equate value [0x" + Long.toHexString(equate.getValue()) + "].");
                    return;
                }
            } else {
                try {
                    equate = this.equateTable.createEquate(equateName, value);
                }
                catch (DuplicateNameException e) {
                    throw new AssertException("Got duplicate name while creating equate " + equateName);
                }
                catch (InvalidInputException e) {
                    this.log.appendMsg("Invalid name for equate " + equateName);
                    return;
                }
            }
            if ((existingEquate = this.equateTable.getEquate(addr, opIndex, equate.getValue())) != null && overwrite) {
                existingEquate.removeReference(addr, opIndex);
            }
            equate.addReference(addr, opIndex);
        }
        catch (Exception e) {
            this.log.appendException((Throwable)e);
        }
    }

    void write(XmlWriter writer, AddressSetView set, TaskMonitor monitor) throws CancelledException {
        Reference ref;
        int i;
        Reference[] refs;
        if (set == null) {
            set = this.program.getMemory();
        }
        writer.startElement("MARKUP");
        monitor.setMessage("Exporting References...");
        AddressIterator iter = this.refManager.getReferenceSourceIterator(set, true);
        while (iter.hasNext()) {
            refs = this.refManager.getReferencesFrom(iter.next());
            for (i = 0; i < refs.length; ++i) {
                if (monitor.isCancelled()) {
                    throw new CancelledException();
                }
                ref = refs[i];
                if (!ref.isMemoryReference()) continue;
                this.writeMemoryReference(ref, writer);
            }
        }
        iter = this.refManager.getReferenceSourceIterator(set, true);
        while (iter.hasNext()) {
            refs = this.refManager.getReferencesFrom(iter.next());
            for (i = 0; i < refs.length; ++i) {
                if (monitor.isCancelled()) {
                    throw new CancelledException();
                }
                ref = refs[i];
                if (!ref.isStackReference()) continue;
                this.writeStackReference((StackReference)ref, writer);
            }
        }
        iter = this.refManager.getReferenceSourceIterator(set, true);
        while (iter.hasNext()) {
            refs = this.refManager.getReferencesFrom(iter.next());
            for (i = 0; i < refs.length; ++i) {
                if (monitor.isCancelled()) {
                    throw new CancelledException();
                }
                ref = refs[i];
                if (!ref.isExternalReference()) continue;
                this.writeExternalReference((ExternalReference)ref, writer);
            }
        }
        this.writeEquateReferences(writer, set, monitor);
        writer.endElement("MARKUP");
    }

    private void writeMemoryReference(Reference ref, XmlWriter writer) {
        if (ref.getSource() == SourceType.USER_DEFINED) {
            XmlAttributes attr = new XmlAttributes();
            this.addCommonRefAttributes(attr, ref);
            attr.addAttribute("TO_ADDRESS", XmlProgramUtilities.toString((Address)ref.getToAddress()));
            if (ref.isOffsetReference()) {
                attr.addAttribute("BASE_ADDRESS", XmlProgramUtilities.toString((Address)((OffsetReference)ref).getBaseAddress()));
            } else if (ref.isShiftedReference()) {
                // empty if block
            }
            attr.addAttribute("PRIMARY", ref.isPrimary());
            writer.writeElement("MEMORY_REFERENCE", attr);
        }
    }

    private void writeStackReference(StackReference ref, XmlWriter writer) {
        if (ref.getSource() == SourceType.USER_DEFINED && ref.isStackReference()) {
            XmlAttributes attr = new XmlAttributes();
            this.addCommonRefAttributes(attr, (Reference)ref);
            attr.addAttribute("STACK_PTR_OFFSET", ref.getStackOffset(), true);
            writer.writeElement("STACK_REFERENCE", attr);
        }
    }

    private void writeExternalReference(ExternalReference ref, XmlWriter writer) {
        XmlAttributes attr = new XmlAttributes();
        this.addCommonRefAttributes(attr, (Reference)ref);
        ExternalLocation extLoc = ref.getExternalLocation();
        attr.addAttribute("LIB_PROG_NAME", extLoc.getLibraryName());
        String label = extLoc.getLabel();
        Address addr = extLoc.getAddress();
        if (label != null) {
            attr.addAttribute("LIB_LABEL", label);
        }
        if (addr != null) {
            attr.addAttribute("LIB_ADDR", XmlProgramUtilities.toString((Address)addr));
        }
        writer.writeElement("EXT_LIBRARY_REFERENCE", attr);
    }

    private void writeEquateReferences(XmlWriter writer, AddressSetView set, TaskMonitor monitor) throws CancelledException {
        monitor.setMessage("Exporting Equate References...");
        Iterator iter = this.equateTable.getEquates();
        while (iter.hasNext()) {
            if (monitor.isCancelled()) {
                throw new CancelledException();
            }
            Equate equate = (Equate)iter.next();
            String name = equate.getName();
            long value = equate.getValue();
            EquateReference[] refs = equate.getReferences();
            for (int i = 0; i < refs.length; ++i) {
                if (monitor.isCancelled()) {
                    return;
                }
                Address addr = refs[i].getAddress();
                if (!set.contains(addr)) continue;
                XmlAttributes attr = new XmlAttributes();
                attr.addAttribute("ADDRESS", XmlProgramUtilities.toString((Address)addr));
                attr.addAttribute("OPERAND_INDEX", refs[i].getOpIndex(), true);
                attr.addAttribute("NAME", name);
                attr.addAttribute("VALUE", value, true);
                writer.writeElement("EQUATE_REFERENCE", attr);
            }
        }
    }

    private void addCommonRefAttributes(XmlAttributes attr, Reference ref) {
        attr.addAttribute("ADDRESS", XmlProgramUtilities.toString((Address)ref.getFromAddress()));
        int opIndex = ref.getOperandIndex();
        if (opIndex != -1) {
            attr.addAttribute("OPERAND_INDEX", opIndex, true);
        }
        attr.addAttribute("USER_DEFINED", ref.getSource() == SourceType.USER_DEFINED);
    }
}

