# Copyright (C) 2025 Intel Corporation
# SPDX-License-Identifier: MIT

import json
from pathlib import Path
from typing import Dict, Union

VERSION_FILE_PATH = Path(__file__).parent / 'mpp_version.json'


KEY_MAJOR = "major_version"
KEY_MINOR = "minor_version"
KEY_PATCH = "patch_version"
KEY_PRE_RELEASE = "pre_release"
KEY_PRE_RELEASE_NUMBER = "pre_release_number"
KEY_BUILD_ID = "build_id"
KEY_BUILD_SHA = "build_sha"


def load_mpp_version_info() -> Dict[str, Union[int, str]]:
    with open(VERSION_FILE_PATH, 'r') as f:
        mpp_version_info = json.load(f)
    return mpp_version_info


class MPPVersionInfo:

    def __init__(self):
        self.__version_info = load_mpp_version_info()

    def __str__(self):
        return self.get_version()

    @property
    def build_id(self):
        return self.__get_build_id()

    def get_version(self) -> str:
        version_str = f'{self.__version_info[KEY_MAJOR]}.{self.__version_info[KEY_MINOR]}.{self.__version_info[KEY_PATCH]}'
        if self.__version_info[KEY_PRE_RELEASE]:
            version_str += f'{self.__version_info[KEY_PRE_RELEASE]}'
            if self.__version_info[KEY_PRE_RELEASE_NUMBER]:
                version_str += f'{self.__version_info[KEY_PRE_RELEASE_NUMBER]}'
        return version_str

    def get_version_and_build_id(self) -> str:
        version_and_build_id = f'{self.get_version()}'
        build_id = self.__get_build_id()
        if build_id:
            version_and_build_id += f' (build {build_id})'
        return version_and_build_id

    def __get_build_id(self) -> int:
        return self.__version_info[KEY_BUILD_ID]

    def __get_build_sha(self) -> int:
        return self.__version_info[KEY_BUILD_SHA]

version_info = MPPVersionInfo()

__version__ = version_info


