#
# Copyright (C) 2017 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#

class InstructionMixAccumulator(object):
    def __init__(self, keys_values):
        self.values = dict(keys_values)
        self.it = iter(sorted(self.values.keys()))

    def add(self, line):
        for key in self.values.keys():
            try:
                self.values[key] += int(line[key])
            except KeyError:
                pass
            except ValueError:
                pass

    def __iter__(self):
        return self.it

    def next(self):
        return next(self.it)

    __next__ = next

    def __getitem__(self, key):
        return str(self.values[key])

    def __str__(self):
        return str(self.values)

    def __repr__(self):
        return repr(self.values)

    def __len__(self):
        return len(self.values)


def _get_children_and_ids(rows_to_check, rows, ids):
    for row in rows_to_check:
        if not row["loop_function_id"] in ids:
            rows.append(row)
            ids.add(row["loop_function_id"])
        (rows, ids) = _get_children_and_ids(row.children, rows, ids)
    return (rows, ids)

def get_static_total_mix(slf, static_type):
    (slf_rows, _) = _get_children_and_ids(slf.sync, [], set([slf["loop_function_id"]]))
    rows_to_aggregate = []
    ids_to_aggregate = set()
    for row in slf_rows:
        for sync_row in row.sync:
            if not sync_row["loop_function_id"] in ids_to_aggregate:
                ids_to_aggregate.add(sync_row["loop_function_id"])
                rows_to_aggregate.append(sync_row)
    base_mix = next(slf.get_instruction_mix(static_type))
    keys_values = [(key, int(base_mix[key])) for key in base_mix]
    accumulator = InstructionMixAccumulator(keys_values)
    for row in rows_to_aggregate:
        accumulator.add(next(row.get_instruction_mix(static_type)))
    return iter([accumulator])


class InstructionMixLine(object):
    def __init__(self, line, prefix=None):
        self.values = dict()
        for key in line:
            new_key = key[len(prefix):] if prefix and key.startswith(prefix) else key
            # Here we provide a default value of "0" for instruction mix metrics
            # to make it easier to use instruction mix in conversion and other formulas.
            self.values[new_key] = line[key] or '0'
        self.it = iter(sorted(self.values.keys()))

    def __iter__(self):
        return self.it

    def next(self):
        return next(self.it)

    __next__ = next

    def __getitem__(self, key):
        return self.values[key]

    def __str__(self):
        return str(self.line)

    def __repr__(self):
        return repr(self.line)

    def __len__(self):
        return len(self.line)


class InstructionMixIterator(object):
    def __init__(self, it, prefix=None):
        self.it = it
        self.prefix = prefix

    def next(self):
        return InstructionMixLine(next(self.it), self.prefix)

    __next__ = next

    def __iter__(self):
        return self

