#
# Copyright (C) 2017 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#


# ------------------------------------------------------------------------------
# This example shows how to obtain assembly information from the bottomup data.
# ------------------------------------------------------------------------------

import sys

try:

    import advisor

except ImportError:

    print(
        """Import error: Python could not resolve path to Advisor's pythonapi directory.
        To fix, either manually add path to the pythonapi directory into PYTHONPATH environment
        variable, or use advixe-vars.* scripts to set up product environment variables automatically."""
    )
    sys.exit(1)

# Check command-line arguments.
if len(sys.argv) < 2:
    print('Usage: "python {} path_to_project_dir [source:line]"'.format(__file__))
    sys.exit(2)

# Open the Advisor Project and load the data.
project = advisor.open_project(sys.argv[1])
data = project.load(advisor.ALL)

source_location = sys.argv[2] if len(sys.argv) > 2 else ""

for entry in data.bottomup:

    if source_location and not source_location in entry["source_location"]:

        continue

    print(entry["mangled_name"])

    # Print assembly for first entry in bottom-up dataset.
    for line in entry.assembly:
        print(line["asm"])

    # Print all assembly data.
    print("\nDetails:")
    for line in entry.assembly:
        for key in [k for k in line if k != "operands"]:
            print("    {}: {}".format(key, line[key]))
        print("    Operands:")
        for operand in line["operands"]:
            for op_detail in operand:
                print("        {}: {}".format(op_detail, operand[op_detail]))
            print("")
        print("-" * 32)

    # Iterating through basic blocks in a loop/function (excluding those that
    # inside inner loop/function).
    for basic_block in entry.basic_blocks:
        print(
            "start_rva {}: call_count {}: total_time {}".format(
                basic_block.start_rva, basic_block.call_count, basic_block.total_time
            )
        )
        for line in basic_block.assembly:
            print(line["asm"])
        for metrics in basic_block.static_mix:
            for metric in metrics:
                print("{}: {}".format(metric, metrics[metric]))
        print("\nOperations:")
        for arith_type in basic_block.operations_metrics:
            print("{}:".format(arith_type))
            for operand_width in basic_block.operations_metrics[arith_type]:
                print("{} bits:".format(operand_width))
                for isa in basic_block.operations_metrics[arith_type][operand_width]:
                    print("    {}: {}".format(isa, basic_block.operations_metrics[arith_type][operand_width][isa],))

    # Break after first found entry - assembly for all binaries could be quite long.
    break
